require 'bmc/ipmi'
require 'bmc/redfish'

module Proxy::BMC
  class Api < ::Sinatra::Base
    helpers ::Proxy::Helpers
    authorize_with_trusted_hosts
    authorize_with_ssl_client
    # All GET requests will only read ipmi data, no changes
    # All PUT requests will update information on the bmc device

    before do
      content_type :json
    end

    # return list of available options
    get "/" do
      res = ['providers', 'providers/installed', 'host']
      { :available_resources => res}.to_json
    end

    # Returns a list of bmc providers
    get "/providers" do
      { :providers => Proxy::BMC::IPMI.providers + non_ipmi_providers }.to_json
    end

    # Returns a list of installed providers
    get "/providers/installed" do
      { :installed_providers => Proxy::BMC::IPMI.providers_installed + non_ipmi_providers }.to_json
    end

    # returns a helpful message that the user should supply a hostname
    get "/host" do
      { :message => "You need to supply the hostname or ip of the actual bmc device"}.to_json
    end

    # returns host operations
    get "/:host" do
      { :actions => %w[chassis lan test fru bmc sensors] }.to_json
    end

    # runs a test against the bmc device and returns true if the connection was successful
    get "/:host/test" do
      bmc_setup
      begin
        result = @bmc.test
        { :action => :test, :result => result }.to_json
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    # returns chassis operations
    get "/:host/chassis" do
      { :actions => %w[power identify config] }.to_json
    end

    # Gets the power status, does not change power
    get "/:host/chassis/power/?:action?" do
      # return hint on valid options
      if params[:action].nil?
        return { :actions => ["on", "off", "status"] }.to_json
      end
      bmc_setup
      begin
        case params[:action]
          when "status"
            { :action => params[:action], :result => @bmc.powerstatus }.to_json
          when "off"
            { :action => params[:action], :result => @bmc.poweroff? }.to_json
          when "on"
            { :action => params[:action], :result => @bmc.poweron? }.to_json
          else
            { :error => "The action: #{params[:action]} is not a valid action" }.to_json
        end
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    get "/:host/lan/?:action?" do
      if params[:action].nil?
        return { :actions => ["ip", "netmask", "mac", "gateway", "snmp", "vlanid", "ipsrc", "print"] }.to_json
      end
      bmc_setup
      begin
        case params[:action]
          when "ip"
            { :action => params[:action], :result => @bmc.ip }.to_json
          when "netmask"
            { :action => params[:action], :result => @bmc.netmask }.to_json
          when "mac"
            { :action => params[:action], :result => @bmc.mac }.to_json
          when "gateway"
            { :action => params[:action], :result => @bmc.gateway }.to_json
          when "snmp"
            { :action => params[:action], :result => @bmc.snmp }.to_json
          when "vlanid"
            { :action => params[:action], :result => @bmc.vlanid }.to_json
          when "ipsrc"
            { :action => params[:action], :result => @bmc.ipsrc }.to_json
          when "print"
            { :action => params[:action], :result => @bmc.lanprint }.to_json
          else
            { :error => "The action: #{params[:action]} is not a valid action" }.to_json
        end
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    get "/:host/chassis/identify/?:action?" do
      # return hint on valid options
      if params[:action].nil?
        return { :actions => ["status"] }.to_json
      end
      bmc_setup
      # determine which function should be executed
      begin
        case params[:action]
          when "status"
            { :action => params[:action], :result => @bmc.identifystatus }.to_json
          else
            { :error => "The action: #{params[:action]} is not a valid action" }.to_json
        end
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    get "/:host/chassis/config/?:function?" do
      # return hint on valid options
      # removing bootdevice until its supported in rubyipmi
      if params[:function].nil?
        # return {:actions => ["bootdevice", "bootdevices"]}.to_json
        return { :functions => ["bootdevices"] }.to_json
      end
      bmc_setup
      begin
        case params[:function]
          # when "bootdevice"
          #  @bmc.chassis.config.bootdevice.to_json
          when "bootdevices"
            { :devices => @bmc.bootdevices }.to_json
          else
            { :error => "The action: #{params[:function]} is not a valid function" }.to_json
        end
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    put "/:host/chassis/power/?:action?" do
      # return hint on valid options
      if params[:action].nil?
        return { :actions => ["on", "off", "cycle", "soft", "reset"] }.to_json
      end
      bmc_setup
      begin
        case params[:action]
          when "on"
            { :action => params[:action], :result => @bmc.poweron }.to_json
          when "off"
            { :action => params[:action], :result => @bmc.poweroff }.to_json
          when "cycle"
            { :action => params[:action], :result => @bmc.powercycle }.to_json
          when "soft"
            { :action => params[:action], :result => @bmc.poweroff(true) }.to_json
          when "reset"
            { :action => params[:action], :result => @bmc.powerreset }.to_json
          else
            { :error => "The action: #{params[:action]} is not a valid action" }.to_json
        end
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    put "/:host/chassis/config/?:function?/?:action?" do
      if params[:function].nil?
        return { :functions => ["bootdevice"] }.to_json
      end
      bmc_setup
      begin
        case params[:function]

          when "bootdevice"
            if params[:action].nil?
              return { :actions => @bmc.bootdevices, :options => ["reboot=true|false", "persistent=true|false"] }.to_json
            end
            case params[:action]
              when /pxe/
                { :action => params[:action], :result => @bmc.bootpxe(params[:reboot], params[:persistent]) }.to_json
              when /cdrom/
                { :action => params[:action], :result => @bmc.bootcdrom(params[:reboot], params[:persistent]) }.to_json
              when /bios/
                { :action => params[:action], :result => @bmc.bootbios(params[:reboot], params[:persistent]) }.to_json
              when /disk/
                { :action => params[:action], :result => @bmc.bootdisk(params[:reboot], params[:persistent]) }.to_json
              else
                # TODO: this appears to be deadcode; the only supported bootdevices are pxe, cdrom, bios, and disk
                if @bmc.bootdevices.include?(params[:action])
                  { :action => params[:action],
                    :result => @bmc.bootdevice = {:device => params[:action], :reboot => params[:reboot],
                                                  :persistent => params[:persistent]},
                  }.to_json
                else
                  { :error => "#{params[:action]} is not a valid boot device" }.to_json
                end
            end
          else
            { :error => "The action: #{params[:function]} is not a valid function" }.to_json
        end
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    put "/:host/chassis/identify/?:action?" do
      if params[:action].nil?
        return { :actions => ["on", "off"] }.to_json
      end
      bmc_setup
      begin
        case params[:action]
          when "on"
            { :action => params[:action], :result => @bmc.identifyon }.to_json
          when "off"
            { :action => params[:action], :result => @bmc.identifyoff }.to_json
          else
            { :error => "The action: #{params[:function]} is not a valid function" }.to_json
        end
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    get "/:host/fru/?:action?" do
      if params[:action].nil?
        return { :actions => %w[list serial manufacturer model asset_tag] }.to_json
      end
      bmc_setup
      begin
        case params[:action]
          when "list"
            { :action => params[:action], :result => @bmc.frulist }.to_json
          when "serial"
            { :action => params[:action], :result => @bmc.serial }.to_json
          when "manufacturer"
            { :action => params[:action], :result => @bmc.manufacturer }.to_json
          when "model"
            { :action => params[:action], :result => @bmc.model }.to_json
          when "asset_tag"
            { :action => params[:action], :result => @bmc.asset_tag}.to_json
          else
            { :error => "The action: #{params[:action]} is not a valid action" }.to_json
        end
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    get "/:host/bmc/?:action?" do
      if params[:action].nil?
        return { :actions => %w[info guid version] }.to_json
      end
      bmc_setup
      begin
        case params[:action]
          when "info"
            { :action => params[:action], :result => @bmc.info }.to_json
          when "guid"
            { :action => params[:action], :result => @bmc.guid }.to_json
          when "version"
            { :action => params[:action], :result => @bmc.version }.to_json
          else
            { :error => "The action: #{params[:action]} is not a valid action" }.to_json
        end
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    put "/:host/bmc/?:action?" do
      if params[:action].nil?
        return { :actions => %w[reset] }.to_json
      end
      bmc_setup
      begin
        case params[:action]
          when "reset"
            if params[:type].nil?
              return { :options => "type=cold|warm" }.to_json
            end
            case params[:type]
              when /cold|warm/
                { :action => params[:action], :result => @bmc.reset(params[:type])}.to_json
              else
                { :error => "The type: #{params[:type]} is not a valid type" }.to_json
            end
          else
            { :error => "The action: #{params[:action]} is not a valid action" }.to_json
        end
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    get "/:host/sensors/?:action?/?:sensor?" do
      if params[:action].nil?
        return { :actions => %w[list count names fanlist templist get] }.to_json
      end
      bmc_setup
      begin
        case params[:action]
          when "list"
            { :action => params[:action], :result => @bmc.sensorlist }.to_json
          when "count"
            { :action => params[:action], :result => @bmc.sensorcount }.to_json
          when "names"
            { :action => params[:action], :result => @bmc.sensornames }.to_json
          when "fanlist"
            { :action => params[:action], :result => @bmc.fanlist }.to_json
          when "templist"
            { :action => params[:action], :result => @bmc.templist }.to_json
          when "get"
            if params[:sensor].nil?
              return { :options => "sensor=<name>" }.to_json
            end
            { :action => params[:action], :result => @bmc.sensorget(params[:sensor]) }.to_json
          else
            { :error => "The action: #{params[:action]} is not a valid action" }.to_json
        end
      rescue NotImplementedError => e
        log_halt 501, e
      rescue => e
        log_halt 400, e
      end
    end

    def non_ipmi_providers
      ['redfish', 'ssh', 'shell']
    end

    # returns a provider type by validating the given type.  If for some reason the type is invalid
    # this function will try and find the first available provider that could be used.  If there are no providers available
    # lets halt and notify the user.
    def find_ipmi_provider(provider_type)
      # check to see if provider is given and if no default provider is set, search for installed providers
      unless Proxy::BMC::IPMI.installed?(provider_type)
        # check if provider_type is a valid type
        if Proxy::BMC::IPMI.providers.include?(provider_type)
          logger.warn "#{provider_type} specified but it is not installed"
        else
          logger.warn "Invalid BMC type: #{provider_type}, must be one of #{Proxy::BMC::IPMI.providers.join(',')}"
        end
        if !Proxy::BMC::IPMI.providers_installed.empty?
          provider_type = Proxy::BMC::IPMI.providers_installed.first
          logger.warn "Using #{provider_type} as the default BMC provider"
        else
          log_halt 400, "No BMC providers are installed, please install at least freeipmi or ipmitool"
        end
      end
      provider_type
    end

    def bmc_setup
      # Either use the default provider or allow user to specify provider in request
      provider_type = params['bmc_provider'] || body_parameters['bmc_provider'] || Proxy::BMC::Plugin.settings.bmc_default_provider
      provider_type&.downcase!

      # unless the provider is shell find a suitable provider
      unless non_ipmi_providers.include? provider_type
        provider_type = find_ipmi_provider(provider_type)
      end

      case provider_type
        when 'freeipmi', 'ipmitool'
          log_halt 401, "unauthorized" unless auth.provided?
          log_halt 401, "bad_authentication_request, credentials are not in auth.basic format" unless auth.basic?
          username, password = auth.credentials
          # this causes rubyipmi to use the supplied logger, most actions in rubyipmi only output during Logger::DEBUG
          Proxy::BMC::IPMI.logger = logger

          # we use the http auth basic header to pass credentials
          # The idea here is to pass the credentials on the command line
          # to execute ipmi commands and has nothing to do with authorization
          # of using smart-proxy. Its simply a tunnel to pass credentials through,
          # since we are essentially remotely executing ipmi commands using Rubyipmi.

          args = {
            :host         => params[:host],
            :username     => username,
            :options      => body_parameters['options'],
            :password     => password,
            :bmc_provider => provider_type,
          }
          @bmc = Proxy::BMC::IPMI.new(args)
        when 'redfish'
          log_halt 401, "unauthorized" unless auth.provided?
          log_halt 401, "bad_authentication_request, credentials are not in auth.basic format" unless auth.basic?
          username, password = auth.credentials

          args = {
            :host     => params[:host],
            :username => username,
            :options  => body_parameters['options'],
            :password => password,
          }
          @bmc = Proxy::BMC::Redfish.new(args)
        when "shell"
          require 'bmc/shell'
          @bmc = Proxy::BMC::Shell.new
        when "ssh"
          require 'bmc/ssh'
          @bmc = Proxy::BMC::SSH.new(params[:host])
        else
          log_halt 400, "Invalid BMC type: #{provider_type}"
      end
    rescue => e
      log_halt 400, e
    end

    # gets any parameters that were parsed into the body
    # we have to parse the JSON body in the request and merge into the params object for later use
    # if the user does not set the content type to application/json we will just assume there is garbage in the body
    # also if the user decides to do http://127.0.0.1/bmc/192.168.1.6/test?bmc_provider=freeipmi as well as pass in
    # a json encode body with the parameters, all of these items will be merged together
    def body_parameters
      @body_parameters ||= parse_json_body.merge(params)
    end

    def auth
      @auth ||= Rack::Auth::Basic::Request.new(request.env)
    end
  end
end
